from time import time
import sys
import itertools as it
from config import chars, RTList, RFList, RTqstart

'''
Copyright Dr Paul Brown and Prof. Trevor Fenner 2020.
'''

def RFHelper(n, q, L):
    # generate canonical weight sequences of the trees with the root removed
    # n is the order, q is the weight of the first child of the root

    # the case q = 2 includes q = 1 when t = 0
    if q == 2:
        for t in range((n - 1) // 2, -1, -1):
            yield '21' * t + '1' * (n - 1 - 2 * t)

    # the next three cases when q >= (n + 1) // 2 are only used
	# in InitialiseRTList and the recursive call of RFHelper
    elif q == n - 1:
        for a in RTList[q]: yield a
    elif q == n - 2:
        for a in RTList[q]: yield a + '1'
    elif q >= (n + 1) // 2:
        for a in RTList[q]:
            for bhash in RFList[n - q]: yield a + bhash

    elif q == (n - 1) // 2:
        start = 0
        # for even n skip trees where the second subtree is of order n / 2
        if n % 2 == 0: start = len(RTList[n // 2])
        for a in RTList[q]:
            for bhash in RFList[n - q][start:]: yield a + bhash
            # increment the start point of RFList to keep a >= bhash
            start +=1

    # canonical weight sequences are cached for orders up to L
    elif q >= n - L:
        for a in RTList[q]:
            # start is the index in RTList[n - q] of the first tree
			# where the weight of the first child = q
            start = RTqstart[n - q][q]
            # use a sentinel to compare the weight sequence
			# of the first subtree with that of the second subtree
            a_sentinel = a + 'z'
			# newstart is the index in RFList[n - q] of the first tree
			# where the second subtree is identical to the first
            for newstart, bhash in enumerate(RFList[n - q][start:], start):
                if a_sentinel >= bhash: break
            for bhash in RFList[n - q][newstart:]: yield a + bhash

    # recursive case
    else:
        # allocate array for generators of the forests (de-rooted trees)
        # of order n - q where the weight of the first child = r
        RFGenArr = [None for _ in range(q + 1)]
        for r in range(2, q + 1):
            RFGenArr[r] = RFHelper(n - q, r, L)
        for a in RTList[q]:
            a_sentinel = a + 'z'
            for r in range(q, 1, - 1):
                # make a copy of the generator for this value of r
                RFGenArr[r], RFHelperListGen = it.tee(RFGenArr[r], 2)
                if r == q:
                    for bhash in RFHelperListGen:
                        if a_sentinel >= bhash: break
                    yield a + bhash
                for bhash in RFHelperListGen: yield a + bhash

def RootedTrees(n, L):
    # generate 2-tuples comprising the canonical weight sequences of the tree
	# and its forest (de-rooted tree) for all rooted trees of order n
    # also initialises RTqstart

    i = 0
    rootchar = chars[n]
    for q in range(n - 1, 1, -1):
        # stores the index of the first tree in RTlist[n]
        # for which the weight of the first child = q
        RTqstart[n][q] = i
        for a in RFHelper(n, q, L):
            yield rootchar + a, a
            i +=1
    # add the index for q = 1 (i.e. the star)
    RTqstart[n][1]= i - 1

def InitialiseRTList(L):
    # compute canonical weight sequences for rooted trees
	# and their forests up to order L

    # generate rooted trees and forests of order > 4
    for k in range(5, L + 1):
        RTTemp = list(RootedTrees(k, L))
        # rooted trees
        RTList[k] = [a[0] for a in RTTemp]
        # forests
        RFList[k] = [a[1] for a in RTTemp]

def UFT(n, L):
    # generate canonical weight sequences of unicentroidal free trees

    rootchar = chars[n]
    for q in range((n - 1) // 2, 1, -1):
        for a in RFHelper(n, q, L): yield (rootchar + a)

def BFT(n):
    # generate canonical weight sequences of bicentroidal free trees

    for start, a1 in enumerate(RTList[n // 2]):
        for a2 in RTList[n // 2][start:]: yield a1 + a2

def FreeTrees(n, L):
    # generate canonical weight sequences of free trees order n

    if n == 1: yield '1'
    elif n == 2: yield '11'
    elif n == 3: yield '311'
    elif n == 4: yield '4111'; yield '2121'
    else:
        for a in UFT(n, L): yield a
        # add bicentroidal trees when n is even
        if n % 2 == 0:
            for a in BFT(n): yield a

def main(N):
    sttime = time()
    count = 0
    L = (N // 2) + 1
    # initialise cache for order <= L
    InitialiseRTList(L)
    # construct weight sequences of free trees of order N
    for g in FreeTrees(N, L):
        # print(g)
        count += 1
    print('BRFE', N, count, time() - sttime)

if __name__ == '__main__':
    # to check orders up to 30
    for i in range(29):
       main(i)
    # to run from command line use:
    # main(int(sys.argv[1]))

